/* SSL support via OpenSSL library.
   Copyright (C) 2000-2005 Free Software Foundation, Inc.
   Originally contributed by Christian Fraenkel.

This file is part of GYachI

GyachI is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

GyachI is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Wget; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

In addition, as a special exception, the copyright holders of this file
give permission to link the code of its release of GyachI with the
OpenSSL project's "OpenSSL" library (or with modified versions of it
that use the same license as the "OpenSSL" library), and distribute
the linked executables.  You must obey the GNU General Public License
in all respects for all of the code used other than "OpenSSL".  If you
modify this file, you may extend this exception to your version of the
file, but you are not obligated to do so.  If you do not wish to do
so, delete this exception statement from your version.  */

#include <config.h>

#include <assert.h>
#include <errno.h>
#include <ctype.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# include <strings.h>
#endif

#include <openssl/ssl.h>
#include <openssl/x509.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#include <stdio.h>

#include <gtk/gtk.h>

#include "config.h"

#include "gyach.h"
#include "interface.h"
#include "util.h"
#include "gyachi_ssl.h"

#include "gyachi_lib.h"


#define PTR_FORMAT(p) 2 * sizeof (void *), (unsigned long) (p)
#define XNUM_TO_DIGIT(x) ("0123456789ABCDEF"[x] + 0)

/* forward declarations */
const char *escnonprint (const char *str);

/* Application-wide SSL context.  This is common to all SSL
   connections.  */
SSL_CTX *ssl_ctx = NULL;

/* Initialize the SSL's PRNG using various methods. */

static void init_prng (void)
{
  char namebuf[256];
  const char *random_file;

  if (RAND_status ())
    /* The PRNG has been seeded; no further action is necessary. */
    return;

  /* Seed from a file specified by the user.  This will be the file
     specified with --random-file, $RANDFILE, if set, or ~/.rnd, if it
     exists.  */
      /* Get the random file name using RAND_file_name. */
  namebuf[0] = '\0';
  random_file = RAND_file_name (namebuf, sizeof (namebuf));

  if (random_file && *random_file)
    /* Seed at most 16k (apparently arbitrary value borrowed from
       curl) from random file. */
    RAND_load_file (random_file, 16384);

  if (RAND_status ())
    return;

  if (RAND_status ())
    return;

#ifdef WINDOWS
  /* Under Windows, we can try to seed the PRNG using screen content.
     This may or may not work, depending on whether we'll calling Wget
     interactively.  */

  RAND_screen ();
  if (RAND_status ())
    return;
#endif

#if 0 /* don't do this by default */
  {
    int maxrand = 500;

    /* Still not random enough, presumably because neither /dev/random
       nor EGD were available.  Try to seed OpenSSL's PRNG with libc
       PRNG.  This is cryptographically weak and defeats the purpose
       of using OpenSSL, which is why it is highly discouraged.  */

    logprintf (LOG_NOTQUIET, _("WARNING: using a weak random seed.\n"));

    while (RAND_status () == 0 && maxrand-- > 0)
      {
	unsigned char rnd = random_number (256);
	RAND_seed (&rnd, sizeof (rnd));
      }
  }
#endif
}

/* Print errors in the OpenSSL error stack. */

static void print_errors (void) 
{
  unsigned long curerr = 0;
  while ((curerr = ERR_get_error ()) != 0) {
    char buff[512];
    snprintf(buff, 511, "%s %s", 
	     _("OpenSSL:"),
	     ERR_error_string(curerr, NULL));
    if (capture_fp) {
      fprintf(capture_fp, "\n[%s] %s\n", gyach_timestamp(), buff);
    }
    show_ok_dialog(buff);
  }
}


/* Create an SSL Context and set default paths etc.  Called the first
   time an HTTP download is attempted.

   Returns SSL_CTX * on success, 0 otherwise.  */

int ssl_init ()
{
  SSL_METHOD *meth;

  if (ssl_ctx)
    /* The SSL has already been initialized. */
    return 1;

  /* Init the PRNG.  If that fails, bail out.  */
  init_prng ();
  if (RAND_status () != 1)
    {
      char buff[512];
      snprintf(buff, 511, "%s %s",
	       _("OpenSSL:"),
	       _("Could not seed PRNG"));
      if (capture_fp) {
	fprintf(capture_fp, "\n[%s] %s\n",
		gyach_timestamp(),
		buff);
      }

      show_ok_dialog(buff);
      goto error;
    }

  SSL_library_init ();
  SSL_load_error_strings ();
  SSLeay_add_all_algorithms ();
  SSLeay_add_ssl_algorithms ();

  //  meth = SSLv23_method ();
  //  meth = SSLv3_client_method ();
  meth = SSLv23_client_method ();

  ssl_ctx = SSL_CTX_new (meth);
  if (!ssl_ctx)
    goto error;

  SSL_CTX_set_default_verify_paths (ssl_ctx);
#if 0
  SSL_CTX_load_verify_locations (ssl_ctx, opt.ca_cert, opt.ca_directory);
#endif

  /* SSL_VERIFY_NONE instructs OpenSSL not to abort SSL_connect if the
     certificate is invalid.  We verify the certificate separately in
     ssl_check_certificate, which provides much better diagnostics
     than examining the error stack after a failed SSL_connect.  */
  SSL_CTX_set_verify (ssl_ctx, SSL_VERIFY_NONE, NULL);

  /* Since fd_write unconditionally assumes partial writes (and
     handles them correctly), allow them in OpenSSL.  */
  SSL_CTX_set_mode (ssl_ctx, SSL_MODE_ENABLE_PARTIAL_WRITE);

  /* The OpenSSL library can handle renegotiations automatically, so
     tell it to do so.  */
  SSL_CTX_set_mode (ssl_ctx, SSL_MODE_AUTO_RETRY);

  SSL_CTX_set_options(ssl_ctx, SSL_OP_NO_SSLv2);

  return 1;

 error:
  if (ssl_ctx) {
    SSL_CTX_free (ssl_ctx);
    ssl_ctx = NULL;
  }
  print_errors ();
  return 0;
}

int openssl_pending(int fd, SSL *ctx)
{
  return(SSL_pending(ctx));
}

int openssl_read(int fd, char *buf, int bufsize, SSL *ctx)
{
  int ret;
  SSL *ssl = (SSL *) ctx;
  do
    ret = SSL_read (ssl, buf, bufsize);
  while (ret == -1
	 && SSL_get_error (ssl, ret) == SSL_ERROR_SYSCALL
	 && errno == EINTR);
  return ret;
}

int openssl_write(int fd, char *buf, int bufsize, SSL *ctx)
{
  int ret = 0;
  SSL *ssl = (SSL *) ctx;
  do
    ret = SSL_write (ssl, buf, bufsize);
  while (ret == -1
	 && SSL_get_error (ssl, ret) == SSL_ERROR_SYSCALL
	 && errno == EINTR);
  return ret;
}

int openssl_poll(int fd, double timeout, int wait_for, SSL *ctx)
{
  SSL *ssl = (SSL *) ctx;
  if (timeout == 0)
    return 1;
  if (SSL_pending (ssl))
    return 1;
  return select_fd(fd, timeout, wait_for);
}

void openssl_close(int fd, SSL *ctx)
{
  SSL *ssl = (SSL *) ctx;
  SSL_shutdown (ssl);
  SSL_free (ssl);

#ifdef WINDOWS
  closesocket (fd);
#else
  close (fd);
#endif

  if (capture_fp) {
    fprintf(capture_fp, "Closed %d/SSL 0x%0lx\n", fd, (unsigned long)ssl);
  }
}

/* Perform the SSL handshake on file descriptor FD, which is assumed
   to be connected to an SSL server.  The SSL handle provided by
   OpenSSL is registered with the file descriptor FD using
   fd_register_transport, so that subsequent calls to fd_read,
   fd_write, etc., will use the corresponding SSL functions.

   Returns 1 on success, 0 on failure.  */

SSL *ssl_connect (int fd) 
{
  SSL *ssl;

  if (capture_fp) {
    fprintf(capture_fp, "Initiating SSL handshake.\n");
  }

  assert(ssl_ctx != NULL);
  ssl = SSL_new(ssl_ctx);
  if (!ssl)
    goto error;
  if (!SSL_set_fd(ssl, fd))
    goto error;
  SSL_set_connect_state(ssl);
  if (SSL_connect (ssl) <= 0 || ssl->state != SSL_ST_OK)
    goto error;

  if (capture_fp) {
    fprintf(capture_fp, "Handshake successful; connected socket %d to SSL handle 0x%0*lx\n",
	    fd, PTR_FORMAT(ssl));
  }
  return ssl;

 error:
  if (capture_fp) {
    fprintf(capture_fp, "SSL handshake failed.\n");
  }
  print_errors();
  if (ssl)
    SSL_free(ssl);
  return 0;
}

#define ASTERISK_EXCLUDES_DOT	/* mandated by rfc2818 */

/* Return 1 is STRING (case-insensitively) matches PATTERN, 0
   otherwise.  The recognized wildcard character is "*", which matches
   any character in STRING except ".".  Any number of the "*" wildcard
   may be present in the pattern.

   This is used to match of hosts as indicated in rfc2818: "Names may
   contain the wildcard character * which is considered to match any
   single domain name component or component fragment. E.g., *.a.com
   matches foo.a.com but not bar.foo.a.com. f*.com matches foo.com but
   not bar.com [or foo.bar.com]."

   If the pattern contain no wildcards, pattern_match(a, b) is
   equivalent to !strcasecmp(a, b).  */

int pattern_match(const char *pattern, const char *string)
{
  const char *p = pattern, *n = string;
  char c;
  for (; (c = tolower(*p++)) != '\0'; n++)
    if (c == '*')
      {
	for (c = tolower(*p); c == '*'; c = tolower(*++p))
	  ;
	for (; *n != '\0'; n++)
	  if (tolower(*n) == c && pattern_match(p, n))
	    return 1;
#ifdef ASTERISK_EXCLUDES_DOT
	  else if (*n == '.')
	    return 0;
#endif
	return c == '\0';
      }
    else
      {
	if (c != tolower(*n))
	  return 0;
      }
  return *n == '\0';
}

/* Verify the validity of the certificate presented by the server.
   Also check that the "common name" of the server, as presented by
   its certificate, corresponds to HOST.  (HOST typically comes from
   the URL and is what the user thinks he's connecting to.)

   This assumes that ssl_connect has successfully finished, i.e. that
   the SSL handshake has been performed and that FD is connected to an
   SSL handle.

   If opt.check_cert is non-zero (the default), this returns 1 if the
   certificate is valid, 0 otherwise.  If opt.check_cert is 0, the
   function always returns 1, but should still be called because it
   warns the user about any problems with the certificate.  */

int ssl_check_certificate (int fd, const char *host, SSL *ctx)
{
  X509 *cert;
  char common_name[256];
  long vresult;
  int success = 1;
  SSL *ssl = (SSL *)ctx;

  assert (ssl != NULL);
  cert = SSL_get_peer_certificate (ssl);
  if (!cert)
    {
      char buff[512];
      snprintf(buff, 511, "%s %s", 
	       _("WARNING: No certificate presented by"),
	       escnonprint(host));
      if (capture_fp) {
	fprintf(capture_fp, "\n[%s] OpenSSL: %s\n",
		gyach_timestamp(),
		buff);
      }
      show_ok_dialog(buff);

      success = 0;
      goto no_cert;		/* must bail out since CERT is NULL */
    }

  if (capture_fp)
    {
      char *subject = X509_NAME_oneline(X509_get_subject_name (cert), 0, 0);
      char *issuer = X509_NAME_oneline(X509_get_issuer_name (cert), 0, 0);

      fprintf(capture_fp, "certificate:\n  subject: %s\n  issuer:  %s\n",
	      escnonprint(subject), escnonprint(issuer));

      OPENSSL_free (subject);
      OPENSSL_free (issuer);
    }

  vresult = SSL_get_verify_result (ssl);
  if (vresult != X509_V_OK)
    {
      /* #### We might want to print saner (and translatable) error
	 messages for several frequently encountered errors.  The
	 candidates would include
	 X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY,
	 X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN,
	 X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT,
	 X509_V_ERR_CERT_NOT_YET_VALID, X509_V_ERR_CERT_HAS_EXPIRED,
	 and possibly others.  The current approach would still be
	 used for the less frequent failure cases.  */

      char buff[512];
      snprintf(buff, 511, "%s %s: %s", 
	       _("OpenSSL: WARNING: Certificate verification error for"),
	       escnonprint(host),
	       X509_verify_cert_error_string (vresult));
      if (capture_fp) {
	fprintf(capture_fp, "\n[%s] %s\n",
		gyach_timestamp(),
		buff);
      }
      show_ok_dialog(buff);

      success = 0;
      /* Fall through, so that the user is warned about *all* issues
	 with the cert (important with --no-check-certificate.)  */
    }

  /* Check that HOST matches the common name in the certificate.
     #### The following remains to be done:

     - It should use dNSName/ipAddress subjectAltName extensions if
       available; according to rfc2818: "If a subjectAltName extension
       of type dNSName is present, that MUST be used as the identity."

     - When matching against common names, it should loop over all
       common names and choose the most specific one, i.e. the last
       one, not the first one, which the current code picks.

     - Ensure that ASN1 strings from the certificate are encoded as
       UTF-8 which can be meaningfully compared to HOST.  */

  common_name[0] = '\0';
  X509_NAME_get_text_by_NID (X509_get_subject_name (cert),
			     NID_commonName, common_name, sizeof (common_name));
  if (!pattern_match(common_name, host))
    {
      char buff[512];
      snprintf(buff, 511, "%s `%s' %s `%s'", 
	       _("WARNING: certificate common name"),
	       escnonprint(common_name),
	       _("doesn't match requested host name:"),
	       escnonprint(host));
      if (capture_fp) {
	fprintf(capture_fp, "\n[%s] OpenSSL: %s\n",
		gyach_timestamp(),
		buff);
      }
      show_ok_dialog(buff);

      success = 0;
    }

  if (success) {
    if (capture_fp) {
      fprintf(capture_fp, "X509 certificate successfully verified and matches host %s\n",
	      escnonprint (host));
    }
  }

  X509_free (cert);

 no_cert:
  {
    char buff[512];
    snprintf(buff, 511, "%s: `%s'", 
	     _("WARNING: Attempting insecure connection to host"),
	     escnonprint(host));
    if (capture_fp) {
      fprintf(capture_fp, "\n[%s] OpenSSL: %s\n",
	      gyach_timestamp(),
	      buff);
    }
    show_ok_dialog(buff);
  }
  return 1;
}



/* String escape functions. */
/* From wget's log.c */
/* Return the number of non-printable characters in SOURCE.
   Non-printable characters are determined as per safe-ctype.c.  */

int count_nonprint (const char *source)
{
  const char *p;
  int cnt;
  for (p = source, cnt = 0; *p; p++)
    if (!isprint(*p))
      ++cnt;
  return cnt;
}

/* Copy SOURCE to DEST, escaping non-printable characters.

   Non-printable refers to anything outside the non-control ASCII
   range (32-126) which means that, for example, CR, LF, and TAB are
   considered non-printable along with ESC, BS, and other control
   chars.  This is by design: it makes sure that messages from remote
   servers cannot be easily used to deceive the users by mimicking
   Wget's output.  Disallowing non-ASCII characters is another
   necessary security measure, which makes sure that remote servers
   cannot garble the screen or guess the local charset and perform
   homographic attacks.

   Of course, the above mandates that escnonprint only be used in
   contexts expected to be ASCII, such as when printing host names,
   URL components, HTTP headers, FTP server messages, and the like.

   ESCAPE is the leading character of the escape sequence.  BASE
   should be the base of the escape sequence, and must be either 8 for
   octal or 16 for hex.

   DEST must point to a location with sufficient room to store an
   encoded version of SOURCE.  */

void copy_and_escape (const char *source, char *dest, char escape, int base)
{
  const char *from = source;
  char *to = dest;
  unsigned char c;

  /* Copy chars from SOURCE to DEST, escaping non-printable ones. */
  switch (base)
    {
    case 8:
      while ((c = *from++) != '\0')
	if (isprint(c))
	  *to++ = c;
	else
	  {
	    *to++ = escape;
	    *to++ = '0' + (c >> 6);
	    *to++ = '0' + ((c >> 3) & 7);
	    *to++ = '0' + (c & 7);
	  }
      break;
    case 16:
      while ((c = *from++) != '\0')
	if (isprint(c))
	  *to++ = c;
	else
	  {
	    *to++ = escape;
	    *to++ = XNUM_TO_DIGIT (c >> 4);
	    *to++ = XNUM_TO_DIGIT (c & 0xf);
	  }
      break;
    default:
      abort ();
    }
  *to = '\0';
}

#define RING_SIZE 3
struct ringel {
  char *buffer;
  int size;
};
static struct ringel ring[RING_SIZE];	/* ring data */

const char *escnonprint_internal (const char *str, char escape, int base)
{
  static int ringpos;		        /* current ring position */
  int nprcnt;

  assert (base == 8 || base == 16);

  nprcnt = count_nonprint (str);
  if (nprcnt == 0)
    /* If there are no non-printable chars in STR, don't bother
       copying anything, just return STR.  */
    return str;

  {
    /* Set up a pointer to the current ring position, so we can write
       simply r->X instead of ring[ringpos].X. */
    struct ringel *r = ring + ringpos;

    /* Every non-printable character is replaced with the escape char
       and three (or two, depending on BASE) *additional* chars.  Size
       must also include the length of the original string and one
       additional char for the terminating \0. */
    int needed_size = strlen (str) + 1 + (base == 8 ? 3 * nprcnt : 2 * nprcnt);

    /* If the current buffer is uninitialized or too small,
       (re)allocate it.  */
    if (r->buffer == NULL || r->size < needed_size)
      {
	r->buffer = realloc(r->buffer, needed_size);
	r->size = needed_size;
      }

    copy_and_escape(str, r->buffer, escape, base);
    ringpos = (ringpos + 1) % RING_SIZE;
    return r->buffer;
  }
}

/* Return a pointer to a static copy of STR with the non-printable
   characters escaped as \ooo.  If there are no non-printable
   characters in STR, STR is returned.  See copy_and_escape for more
   information on which characters are considered non-printable.

   DON'T call this function on translated strings because escaping
   will break them.  Don't call it on literal strings from the source,
   which are by definition trusted.  If newlines are allowed in the
   string, escape and print it line by line because escaping the whole
   string will convert newlines to \012.  (This is so that expectedly
   single-line messages cannot use embedded newlines to mimic Wget's
   output and deceive the user.)

   escnonprint doesn't quote its escape character because it is notf
   meant as a general and reversible quoting mechanism, but as a quick
   way to defang binary junk sent by malicious or buggy servers.

   NOTE: since this function can return a pointer to static data, be
   careful to copy its result before calling it again.  However, to be
   more useful with printf, it maintains an internal ring of static
   buffers to return.  Currently the ring size is 3, which means you
   can print up to three values in the same printf; if more is needed,
   bump RING_SIZE.  */

const char *escnonprint (const char *str)
{
  return escnonprint_internal (str, '\\', 8);
}

